/*
 * Routines for handling proxy communication with FMS
 */
#include <stdio.h>
#include <netinet/in.h>

#include "libfma.h"
#include "lf_scheduler.h"
#include "lf_fabric.h"
#include "lf_fma_comm.h"
#include "lf_fms_comm.h"
#include "lf_fma_flags.h"

#include "fma.h"
#include "fma_myri_packet.h"
#include "fma_proxy.h"
#include "fma_myri.h"
#include "fma_fms.h"
#include "fma_map.h"
#include "fma_standalone.h"

/*
 * local prototypes
 */
static void fma_send_proxy_client_id_request(struct fma_proxy *pp);
static void fma_proxy_client_id_timeout(void *v);

/*
 * Start using a remote FMA as a proxy for communicating with the FMS.  First
 * we need to request an ID from the guy we hope to use as a proxy.
 * XXX - a good thing to do might be, after we get first and every Nth map,
 * to select a new proxy FMA.  Just call "fma_change_proxy()", it puts us back
 * in pending state and requests a new client ID from someone at random.
 */
void
fma_proxy_initiate(
  struct fma_nic_info *nip,
  int port,
  unsigned char *route,
  int route_len)
{
  struct fma_proxy *pp;

  return;	/* XXX disable */

  /* Should not already be in proxy mode */
  if (A.proxy == NULL) {
  
    /* allocate proxy struct */
    LF_CALLOC(pp, struct fma_proxy, 1);
    A.proxy = pp;
  }

  /* Fill in proxy information */
  pp->nip = nip;
  pp->port = port;
  memcpy(pp->route, route, route_len);
  pp->route_len = route_len;

  /* request proxy client ID */
  fma_send_proxy_client_id_request(pp);

  /* run state is that we are awaiting a proxy client ID */
  A.run_state = FMA_RUN_PROXY_PENDING;

  /* set a timer for getting the proxy client ID */
  pp->timer = lf_schedule_event(fma_proxy_client_id_timeout, NULL,
      		FMA_PROXY_CLIENT_ID_TIMEOUT);
  if (pp->timer == NULL) LF_ERROR(("Error scheduling proxy client ID timeout"));

  /* clean up any ongoing standalone stuff */
  fma_cancel_standalone_mode();

  return;

 except:
  fma_perror_exit(1);
}

/*
 * Send a request for a proxy client ID
 */
static void
fma_send_proxy_client_id_request(
  struct fma_proxy *pp)
{
  struct fma_myri_packet pkt;

  if (A.debug) fma_log("Sending request for proxy client ID");

  pkt.h.type_16 = htons(FMA_PACKET_TYPE);
  pkt.h.subtype_16 = htons(FMA_SUBTYPE_PROXY_CLIENT_ID_REQUEST);

  /* Put return route in request packet */
  lf_reverse_route(pkt.u.proxy_client_id_request.reply_route,
                   pp->route, pp->route_len);
  pkt.u.proxy_client_id_request.reply_route_len_8 = pp->route_len;

  /* send request */
  fma_myri_raw_send(pp->nip->nic_handle, pp->port,
      pp->route, pp->route_len, &pkt, sizeof(pkt), NULL, NULL);

  return;
}

/*
 * Got a proxy client ID reply
 */
void
fma_got_proxy_client_id_reply(
  struct fma_myri_packet *pkt)
{
  struct fma_proxy *pp;
  int rc;

  /* Ignore stale replies */
  if (A.run_state != FMA_RUN_PROXY_PENDING) {
    if (A.debug) fma_log("Ignoring stale proxt client ID reply");
    return;
  }

  pp = A.proxy;

  /* clear timeout timer */
  lf_remove_event(pp->timer);
  pp->timer = NULL;

  /* save the client ID */
  pp->client_id = ntohl(pkt->u.proxy_client_id_reply.client_id_32);

  /* Set proxy mode */
  if (A.debug) fma_log("enter proxy mode, ID=%d", pp->client_id);

  /* set run state and mapper level */
  A.run_state = FMA_RUN_FMS_VIA_PROXY;
  fma_set_mapping_level(FMA_PROXY_MAPPER_LEVEL);

  /* send startup messages */
  rc = fma_send_fma_version();
  if (rc == -1) LF_ERROR(("Error sending FMA version"));
  rc = fma_send_myri_info();
  if (rc == -1) LF_ERROR(("Error sending Myrinet info"));
  return;

 except:
  fma_perror_exit(1);
}

/*
 * Request for proxy client ID was not answered in time.  Revert to standalone
 * mode.
 */
static void
fma_proxy_client_id_timeout(
  void *v)
{
  struct fma_proxy *pp;

  if (A.debug) fma_log("Timed out waiting for proxy client ID");

  pp = A.proxy;
  pp->timer = NULL;		/* timer popped */

  if (A.run_state != FMA_RUN_PROXY_PENDING) {
    LF_ERROR(("Proxy client ID timeout, but not pending!"));
  }

  /* go to standalone mode */
  fma_enter_standalone_mode();

  return;

 except:
  fma_perror_exit(1);
}

/*
 * Cancel any proxy run state "stuff", since we're about to go into some other
 * run state, most likely direct to FMS.
 */
void
fma_cancel_proxy_mode()
{
  struct fma_proxy *pp;

  pp = A.proxy;

  /* cancel any client ID request in progress */
  if (pp->timer != NULL) {
    lf_remove_event(pp->timer);
    pp->timer = NULL;
  }
}

/*
 * Respond to a proxy client ID request
 */
void
fma_got_proxy_client_id_request(
  struct fma_nic_info *nip,
  int port,
  struct fma_myri_packet *pkt)
{
  struct fma_myri_packet reply;
  unsigned char *route;
  int route_len;
  static int next_proxy_client_id = 1;

  if (A.debug) fma_log("Assigning proxy client ID %d", next_proxy_client_id);

  /* start building reply */
  reply.h.type_16 = htons(FMA_PACKET_TYPE);
  reply.h.subtype_16 = htons(FMA_SUBTYPE_PROXY_CLIENT_ID_REPLY);

  /* return next client ID */
  reply.u.proxy_client_id_reply.client_id_32 = htonl(next_proxy_client_id);
  ++next_proxy_client_id;

  /* send request */
  route = pkt->u.proxy_client_id_request.reply_route;
  route_len = pkt->u.proxy_client_id_request.reply_route_len_8;

  fma_myri_raw_send(nip->nic_handle, port, route, route_len,
      &reply, sizeof(reply), NULL, NULL);
}
